# Copyright (C) 2021-2025, dilemma-vr.games
# Shared under MIT license

"""Model module in an MVC design"""

from enum import IntEnum
from PyQt5.QtCore import QObject, pyqtSignal

modes = IntEnum("mode", ["manual", "remote"])
link = IntEnum("link", ["inactive", "started", "locked", "released", "error"])

class Model(QObject):
    """The model contains the complete state of the application"""

    # Signals that allow other parts of the app to listen for model changes
    device_id_changed = pyqtSignal(str)
    mode_changed = pyqtSignal(int)
    tray_open_changed = pyqtSignal(bool)
    drives_changed = pyqtSignal(list)
    selected_drive_changed = pyqtSignal(int)
    active_changed = pyqtSignal(bool)
    link_changed = pyqtSignal(int)
    game_id_changed = pyqtSignal(str)

    def __init__(self):
        """Create a new model object"""
        super().__init__()
        self.device_id = ""
        self.mode = modes.remote
        self.tray_open = False
        self.drives = []
        self.selected_drive = -1
        self.active = False
        self.link = link.inactive
        self.game_id = ""

    @property
    def device_id(self):
        """Device ID generated by dilemma-vr.games and entered by the user"""
        return self._device_id

    @device_id.setter
    def device_id(self, value):
        self._device_id = value
        self.device_id_changed.emit(value)

    @property
    def mode(self):
        """Operation mode: manually/user controller ore remote controlled"""
        return self._mode

    @mode.setter
    def mode(self, value):
        self._mode = value
        self.mode_changed.emit(int(value))

    @property
    def tray_open(self):
        """State of the CD tray"""
        return self._tray_open

    @tray_open.setter
    def tray_open(self, value):
        self._tray_open = value
        self.tray_open_changed.emit(value)
        
    @property
    def drives(self):
        """List of available CD drives"""
        return self._drives

    @drives.setter
    def drives(self, value):
        self._drives = value
        print(value)
        self.drives_changed.emit(value)

    @property
    def selected_drive(self):
        """Index of the selected CD drive"""
        return self._selected_drive

    @selected_drive.setter
    def selected_drive(self, value):
        self._selected_drive = value
        self.selected_drive_changed.emit(value)

    @property
    def active(self):
        """Whether the remote connection is enabled"""
        return self._active

    @active.setter
    def active(self, value):
        self._active = value
        self.active_changed.emit(value)

    @property
    def link(self):
        """Status of the remote connection"""
        return self._link

    @link.setter
    def link(self, value):
        self._link = value
        self.link_changed.emit(int(value))

    @property
    def game_id(self):
        """Game ID as reported by dilemma-vr.games API"""
        return self._game_id

    @game_id.setter
    def game_id(self, value):
        self._game_id = value
        self.game_id_changed.emit(value)

