# Copyright (C) 2021-2025, dilemma-vr.games
# Shared under MIT license

"""View module in MVC design"""

from PyQt5 import QtWidgets, QtCore, QtGui
from PyQt5.QtCore import pyqtSlot, pyqtSignal, Qt

from .model import modes, link

class QClickableLabel(QtWidgets.QLabel):
    """Clickable label for URL"""
    clicked = pyqtSignal()

    def mousePressEvent(self, event):
        self.clicked.emit()
        
class Ui_MainWindow(object):
    """Main window"""

    def setupUi(self, MainWindow):
        """Create all components and connect to view"""

        MainWindow.setObjectName("MainWindow")
        MainWindow.resize(400, 400)
        self.centralwidget = QtWidgets.QWidget(MainWindow)
        self.centralwidget.setObjectName("centralwidget")
        self.vboxlayout = QtWidgets.QVBoxLayout(self.centralwidget)
        self.vboxlayout.setObjectName("vboxlayout")

        self.label_logo = QtWidgets.QLabel("Dilemma Warden")
        self.label_logo.setFont(QtGui.QFont('Arial', 18, QtGui.QFont.Bold))
        self.vboxlayout.addWidget(self.label_logo)

        self.label_link = QClickableLabel("dilemma-vr.games")
        self.label_link.setAlignment(Qt.AlignRight)
        self.label_link.setStyleSheet("QLabel{color : blue; text-decoration: underline}");
        self.label_link.setObjectName("label_link")
        self.vboxlayout.addWidget(self.label_link)

        self.label_desc = QtWidgets.QLabel(
            "Connect your CD tray to the cell door of a Dilemma VR game. "
            "(Depending on your hardware, closing the tray might not work.)"
        )
        self.label_desc.setWordWrap(True)
        self.vboxlayout.addWidget(self.label_desc)

        self.hboxlayout = QtWidgets.QHBoxLayout()
        self.hboxlayout.setObjectName("hboxlayout")
        self.vboxlayout.addLayout(self.hboxlayout)

        self.label_drive = QtWidgets.QLabel("Drive")
        self.label_drive.setObjectName("label_drive")
        self.hboxlayout.addWidget(self.label_drive)

        self.comboBox_drive = QtWidgets.QComboBox()
        self.comboBox_drive.setObjectName("comboBox_drive")
        self.hboxlayout.addWidget(self.comboBox_drive)

        self.label_pos = QtWidgets.QLabel()
        self.label_pos.setObjectName("label_pos")
        self.label_pos.setAlignment(Qt.AlignRight)
        self.vboxlayout.addWidget(self.label_pos)

        self.radio_manual = QtWidgets.QRadioButton("Manual")
        self.radio_manual.setObjectName("radio_manual")
        self.vboxlayout.addWidget(self.radio_manual)

        self.pushButton_toggle = QtWidgets.QPushButton("Open")
        self.pushButton_toggle.setObjectName("pushButton_toggle")
        self.vboxlayout.addWidget(self.pushButton_toggle)

        self.radio_remote = QtWidgets.QRadioButton("Remote controlled")
        self.radio_remote.setObjectName("radio_remote")
        self.vboxlayout.addWidget(self.radio_remote)

        self.hboxlayout2 = QtWidgets.QHBoxLayout()
        self.hboxlayout2.setObjectName("hboxlayout2")
        self.vboxlayout.addLayout(self.hboxlayout2)

        self.label_device_id = QtWidgets.QLabel("Device ID")
        self.label_device_id.setObjectName("label_device_id")
        self.hboxlayout2.addWidget(self.label_device_id)

        self.lineEdit_device_id = QtWidgets.QLineEdit()
        self.lineEdit_device_id.setObjectName("lineEdit_device_id")
        self.hboxlayout2.addWidget(self.lineEdit_device_id)

        self.pushButton_connect = QtWidgets.QPushButton("Connect")
        self.pushButton_connect.setObjectName("pushButton_connect")
        self.vboxlayout.addWidget(self.pushButton_connect)

        self.label_status = QtWidgets.QLabel("Status: not connected")
        self.label_status.setObjectName("label_status")
        self.vboxlayout.addWidget(self.label_status)

        self.label_game_id = QtWidgets.QLabel("Game id: ")
        self.label_game_id.setObjectName("label_game_id")
        self.vboxlayout.addWidget(self.label_game_id)

        MainWindow.setCentralWidget(self.centralwidget)
        MainWindow.setWindowTitle("Dilemma Warden")

        self.retranslateUi(MainWindow)
        QtCore.QMetaObject.connectSlotsByName(MainWindow)

    def retranslateUi(self, MainWindow):
        pass


class MainView(QtWidgets.QMainWindow):
    """View class to connect model and UI components"""

    def __init__(self, model, main_controller):
        super().__init__()

        self._model = model
        self._main_controller = main_controller
        self._ui = Ui_MainWindow()
        self._ui.setupUi(self)

        # connect widgets to controller
        self._ui.lineEdit_device_id.textChanged.connect(self._main_controller.change_device_id)
        self._ui.radio_manual.clicked.connect(
            lambda: self._main_controller.change_mode(modes.manual)
        )
        self._ui.radio_remote.clicked.connect(
            lambda: self._main_controller.change_mode(modes.remote)
        )
        self._ui.comboBox_drive.currentIndexChanged.connect(self._main_controller.change_selected_drive)
        self._ui.pushButton_toggle.clicked.connect(self._main_controller.manual_toggle)
        self._ui.pushButton_connect.clicked.connect(self._main_controller.remote_toggle)
        self._ui.label_link.clicked.connect(self.link_clicked)
        self._ui.lineEdit_device_id.setText(self._model.device_id)

        # listen for model event signals
        self._model.device_id_changed.connect(self.on_device_id_changed)
        self._model.mode_changed.connect(self.on_mode_changed)
        self._model.tray_open_changed.connect(self.on_tray_open_changed)
        self._model.drives_changed.connect(self.on_drives_changed)
        self._model.selected_drive_changed.connect(self.on_selected_drive_changed)
        self._model.active_changed.connect(self.on_active_changed)
        self._model.link_changed.connect(self.on_link_changed)
        self._model.game_id_changed.connect(self.on_game_id_changed)

    @pyqtSlot()
    def link_clicked(self):
        """Open brower"""
        url = QtCore.QUrl("https://dilemma-vr.games")
        QtGui.QDesktopServices.openUrl(url)

    @pyqtSlot(str)
    def on_device_id_changed(self, value):
        currentText = self._ui.lineEdit_device_id.text()
        if currentText != value:
            self._ui.lineEdit_device_id.setText(value)

    @pyqtSlot(int)
    def on_mode_changed(self, value):
        if value == modes.manual:
            self._ui.pushButton_toggle.setEnabled(True)
            self._ui.radio_manual.setChecked(True)
            self._ui.lineEdit_device_id.setEnabled(False)
            self._ui.pushButton_connect.setEnabled(False)
            self._ui.radio_remote.setChecked(False)
            self._ui.label_game_id.setEnabled(False)
            self._ui.label_status.setEnabled(False)
        if value == modes.remote:
            self._ui.pushButton_toggle.setEnabled(False)
            self._ui.radio_manual.setChecked(False)
            self._ui.lineEdit_device_id.setEnabled(not self._model.active)
            self._ui.pushButton_connect.setEnabled(True)
            self._ui.radio_remote.setChecked(True)
            self._ui.label_game_id.setEnabled(True)
            self._ui.label_status.setEnabled(True)

    @pyqtSlot(bool)
    def on_tray_open_changed(self, value):
        self._ui.label_pos.setText("open" if value else "closed")
        self._ui.pushButton_toggle.setText("Close" if value else "Open")

    @pyqtSlot(list)
    def on_drives_changed(self, value):
        self._ui.comboBox_drive.clear()
        for v in value:
            self._ui.comboBox_drive.addItem(v)
        self._ui.comboBox_drive.setCurrentIndex(self._model.selected_drive)

    @pyqtSlot(int)
    def on_selected_drive_changed(self, value):
        self._ui.comboBox_drive.setCurrentIndex(value)

    @pyqtSlot(bool)
    def on_active_changed(self, value):
        self._ui.pushButton_connect.setText("Disconnect" if value else "Connect")
        self._ui.lineEdit_device_id.setEnabled(not value)
        self._ui.radio_manual.setEnabled(not value)

    @pyqtSlot(int)
    def on_link_changed(self, value):
        label = {
            link.inactive: "not connected",
            link.started: "started",
            link.locked: "cell locked",
            link.released: "released",
            link.error: "network error",
        }
        self._ui.label_status.setText("Status: " + label[value])

    @pyqtSlot(str)
    def on_game_id_changed(self, value):
        self._ui.label_game_id.setText("Game ID: " + value)
